
local const = {
    item_prefix = "packing-tape-",
    item_prefix_pattern = "^packing%-tape%-",
    shortcut = "packing-tape-shortcut",
    chest_types = { -- the main inventory of each chest type
        ['container'] = defines.inventory.chest,
        ['logistic-container'] = defines.inventory.chest,
        ['cargo-wagon'] = defines.inventory.cargo_wagon,
        ['car'] = defines.inventory.car_trunk,
        ["spider-vehicle"] = defines.inventory.car_trunk,
    },
    tank_types = {
        ["storage-tank"] = true,
        ["fluid-wagon"] = true,
    },
    entity_filters = {
        {filter="type",type="container"},
        {filter="type",type="logistic-container"},
        {filter="type",type="cargo-wagon"},
        {filter="type",type="car"},
        {filter="type",type="storage-tank"},
        {filter="type",type="fluid-wagon"},
        {filter="type",type="spider-vehicle"},
        {filter="name",name="railloader-placement-proxy"},
        {filter="name",name="railunloader-placement-proxy"},
    },
    mod_entities = {
        ["railloader-placement-proxy"] = "railloader-chest",
        ["railunloader-placement-proxy"] = "railunloader-chest",
        ["supply-depot"] = "supply-depot-chest"
    }
}

-- in case anyone is still using the old filters format
local function filters_deprecated(inventory, filters)
	for index,filter in pairs(filters) do
		inventory.set_filter(index,filter)
	end
end


-- Serializes an equipment grid into a Lua table. Returns an empty table on nil
-- @param grid the equipment grid to serialize
-- @return a table where each extry is data on an equipment
local function serialize_equipment_grid(grid)
	local output = {}
	if grid then
		for i,equipment in pairs(grid.equipment) do
			output[i] = {name = equipment.name,position=equipment.position}
		end
	end
	return output
end

-- Deserializes a grid from a table back into an actual game object
-- @param grid the grid to turn into the listed grid
-- @param the table to put into the grid
local function deserialize_equipment_grid(grid, data)
	if grid and data then
		for _,entry in pairs(data) do
			local equipment = grid.put{name=entry.name,position=entry.position}
		end
	end
end


-- returns the first empty item stack in this inventory that the given item
-- can be inserted into or nil if it's full
-- NOTE: This inventory slot IS NOT valid_for_read
local function find_empty_item_stack(inventory,item_spec)
	if inventory.can_insert(item_spec) then
		for i = 1,#inventory do
			if not inventory[i].valid_for_read and inventory[i].can_set_stack(item_spec) then
				return inventory[i]
			end
		end
	end
	return nil
end

-- transfers each slot in source into its coresponding slot in destination
local function transfer_inventory(source, destination)
    local filtered = source.is_filtered()
    local destination_filterable = destination.supports_filters()
    for i = 1, math.min(#destination, #source) do
        destination[i].transfer_stack(source[i])
        if filtered and destination_filterable then
            destination.set_filter(i, source.get_filter(i))
        end
    end
end

-- inserts each item in source into destination
local function insert_inventory(source, destination)
    if not (source and destination) then return false end
    for i = 1,#source do
        destination.insert(source[i])
    end
    return true
end
-- transfers items from a dictionary of items (like that returned by get_contents) into an inventory
local function transfer_simple_inventory(destination, source)
    if destination and source then
        for name,amount in pairs(source) do
            destination.insert{name=name,count=amount}
        end
    end
end

local function get_spider_remotes(inventory, spidertron)
    if not (inventory and spidertron) then return nil end
    local remotes = {}
    for i = 1,#inventory do
        local item_stack = inventory[i]
        if item_stack.valid_for_read and item_stack.type == "spidertron-remote" then
            local connected_entity = item_stack.connected_entity
            if connected_entity and connected_entity == spidertron then
                remotes[item_stack.item_number] = item_stack.item_number
            end
        end
    end
    return remotes
end

local function set_spider_remotes(inventory, spidertron, remotes)
    if not (inventory and spidertron and remotes) then return nil end
    for i = 1,#inventory do
        local item_stack = inventory[i]
        if item_stack.valid_for_read and item_stack.type == "spidertron-remote" then
            if remotes[item_stack.item_number] and not item_stack.connected_entity then
                item_stack.connected_entity = spidertron
            end
        end
    end
end

local function item_to_container(item, chest, player)
    if not item or not chest then
        log("Missing chest or item when trying to transfer item to chest")
        return
    end
    local item_inventory = item.get_inventory(defines.inventory.item_main)
    local chest_inventory = chest.get_inventory(const.chest_types[chest.type])

    if item_inventory and chest_inventory then
        transfer_inventory(item_inventory, chest_inventory)
    end
    
    local data = global.items[item.item_number]
    if data then
        if data.filters then -- deprecated filter method
            filters_deprecated(item_inventory, data.filters)
        end
        if data.bar then 
            chest_inventory.set_bar(data.bar)
        end
        local proto = chest.prototype
        if proto.logistic_mode == 'storage' then
            chest.storage_filter = data.storage_filter
        elseif proto.logistic_mode == 'requester' or proto.logistic_mode == 'buffer' then
            for slot, filter in pairs(data.item_filter or {}) do
                chest.set_request_slot(filter, slot)
            end
            chest.request_from_buffers = data.request_from_buffers
        end
        if chest.type == "spider-vehicle" then
            local ammo_inv = chest.get_inventory(defines.inventory.car_ammo)
            transfer_simple_inventory(ammo_inv, data.ammo)
        end
        if chest.type == "car" then
            local ammo_inv = chest.get_inventory(defines.inventory.car_ammo)
            transfer_simple_inventory(ammo_inv, data.ammo)
            local fuel_inv = chest.get_inventory(defines.inventory.fuel)
            transfer_simple_inventory(fuel_inv, data.fuel)
            if data.fuel_burning and data.fuel_remaining and chest.burner  then
                chest.burner.currently_burning = data.fuel_burning
                chest.burner.remaining_burning_fuel = data.fuel_remaining
            end
        end
        if data.grid then
            deserialize_equipment_grid(chest.grid,data.grid)
        end
        if data.spider_remotes then
            local player_inventory = player and player.get_main_inventory()
            set_spider_remotes(player_inventory, chest, data.spider_remotes)
        end
    end
    global.items[item.item_number] = nil
end

local function container_to_item(chest, item, player)
    if not chest or not item then
        log("Missing chest or item when trying to transfer chest to item")
        return
    end
    -- Set health in case the chest is pre-damaged
    item.health = chest.get_health_ratio()
    local proto = chest.prototype


    local chest_inventory = chest.get_inventory(const.chest_types[chest.type])
    local item_inventory = item.get_inventory(defines.inventory.item_main)
    transfer_inventory(chest_inventory, item_inventory)

    -- Low overhead call
    global.items = global.items or {}
    global.items[item.item_number] = {}
    local data = global.items[item.item_number]

    data.bar = chest_inventory.supports_bar() and chest_inventory.get_bar()
    data.storage_filter = proto.logistic_mode == 'storage' and chest.storage_filter
    if proto.logistic_mode == 'requester' or proto.logistic_mode == 'buffer' or proto.type == "spider-vehicle" then
        local rsc = chest.request_slot_count
        if rsc > 0 then
            data.item_filter = {}
            for i = 1, chest.request_slot_count do
                data.item_filter[i] = chest.get_request_slot(i)
            end
            data.request_from_buffers = chest.request_from_buffers
        end
    end
    if chest.type == "car" then
        local ammo_inv = chest.get_inventory(defines.inventory.car_ammo)
        data.ammo = ammo_inv and ammo_inv.get_contents()
        local fuel_inv = chest.get_inventory(defines.inventory.fuel)
        data.fuel = fuel_inv and fuel_inv.get_contents()
        if chest.burner then
            data.fuel_burning = chest.burner.currently_burning
            data.fuel_remaining = chest.burner.remaining_burning_fuel
        end
    end
    if chest.type == "spider-vehicle" then
        local ammo_inv = chest.get_inventory(defines.inventory.car_ammo)
        local player_inventory = player and player.get_main_inventory()
        --local player_trash = player and player.character and player.character.get_inventory(defines.inventory.character_trash)
        data.ammo = ammo_inv and ammo_inv.get_contents()
        data.spider_remotes = get_spider_remotes(player_inventory, chest)
        insert_inventory(chest.get_inventory(defines.inventory.spider_trash), player_inventory)
    end
    data.grid = serialize_equipment_grid(chest.grid)
    chest.destroy{raise_destroy=true}
end

local function tank_to_item(tank, item)
    if not tank or not item then
        log("Missing tank or item when trying to transfer tank to item")
        return
    end
    -- Set health in case the chest is pre-damaged
    item.health = tank.get_health_ratio()
    
    local fluidbox = tank.fluidbox and tank.fluidbox[1]
    if fluidbox then 
        item.tags = fluidbox
        item.custom_description = {"item-description.packing-tape-custom",{"fluid-name."..fluidbox.name},fluidbox.amount,fluidbox.temperature}
    end
    tank.destroy{raise_destroy=true}
end

local function item_to_tank(item, tank)
    if not tank or not item then
        log("Missing tank or item when trying to transfer item to tank")
        return
    end

    local fluid = item.tags
    if fluid and fluid.name and fluid.amount then
        tank.insert_fluid(fluid)
    else
        log("missing fluid data in " .. tank.name)
    end
end


local function move_to_container(event)
    
    --[[local chest = event.created_entity

    local entities = game.surfaces[1].find_entities_filtered{position=event.created_entity.position}
    for _,ent in pairs(entities) do
        game.print(ent.name)
    end]]
    
    local stack = event.stack
    local player = game.get_player(event.player_index)
    if stack and stack.valid_for_read and stack.name:find(const.item_prefix_pattern) then
        local chest = event.created_entity
        if chest and const.mod_entities[chest.name]  then
            chest = chest.surface.find_entity(const.mod_entities[chest.name],chest.position)
        end
        if chest then
            if stack.type == "item-with-inventory" then
                item_to_container(stack, chest, player)
            else
                item_to_tank(stack,chest)
            end
        end
    end
end

-- Move the contents from the chest into an item in our inventory
local function move_to_inventory(event)
    local chest = event.entity
    local item_name = const.item_prefix .. chest.name
    if const.chest_types[chest.type] and game.item_prototypes[item_name] then
        if chest.has_items_inside() then
            local player = game.get_player(event.player_index)
            local p_inv = player.get_main_inventory()

            -- Create an item-with-inventory in an available slot
            local stack = find_empty_item_stack(p_inv, item_name)
            -- Should have stack since we can insert but check anyway.
            if player.is_shortcut_toggled(const.shortcut) and stack and stack.set_stack({name=item_name, count=1}) then
                container_to_item(chest, stack, player)
            end
        end
    elseif const.tank_types[chest.type] and game.item_prototypes[item_name] then
        if chest.get_fluid_count() > 0 then
            local player = game.get_player(event.player_index)
            local p_inv = player.get_main_inventory()

            -- Create an item-with-inventory in an available slot
            local stack = find_empty_item_stack(p_inv, item_name)
            -- Should have stack since we can insert but check anyway.
            if player.is_shortcut_toggled(const.shortcut) and stack and stack.set_stack({name=item_name, count=1}) then
                tank_to_item(chest, stack)
            end
        end
    end
end

function toggle_shortcut(event)
    if event.prototype_name == nil or event.prototype_name == const.shortcut then
        local player = game.get_player(event.player_index)
        player.set_shortcut_toggled(const.shortcut, not player.is_shortcut_toggled(const.shortcut))
    end
end

function unpack_item(player, item_stack)
    if item_stack.get_inventory(defines.inventory.item_main) and item_stack.get_inventory(defines.inventory.item_main).is_empty() then
        local item_prototype = item_stack.prototype
        local entity_prototype = item_prototype.place_result
        if entity_prototype and entity_prototype.type ~= "car" then -- ignore cars, they only stack to 1 anyway and have other data stored with them
            local target_items = entity_prototype.items_to_place_this --find the items that normally place this item (it should hopefully not be this item again)
            if target_items and target_items[1] then
                global.items[item_stack.item_number] = nil --remove the item from global to prevent data leaks
                item_stack.clear() --remove the item first to ensure space in the inventory for the new item
                player.insert(target_items[1])
            end            
        end
    end
end

function robot_mined_chest(event)
    local chest = event.entity
    local item_name = const.item_prefix .. chest.name
    if chest and const.chest_types[chest.type] and game.item_prototypes[item_name] then
        if global.decon_chests[chest.unit_number] then
            global.decon_chests[chest.unit_number] = nil
            if chest.has_items_inside() then
                local robot_inventory = event.robot.get_inventory(defines.inventory.robot_cargo)
                local stack = robot_inventory[1]
                if stack and stack.set_stack({name=item_name, count=1}) then
                    container_to_item(chest, stack)
                end
            end
        end
    elseif chest and const.tank_types[chest.type] and game.item_prototypes[item_name] then 
        if global.decon_chests[chest.unit_number] then
            global.decon_chests[chest.unit_number] = nil
            if chest.get_fluid_count() > 0 then
                local robot_inventory = event.robot.get_inventory(defines.inventory.robot_cargo)
                local stack = robot_inventory[1]
                if stack and stack.set_stack({name=item_name, count=1}) then
                    tank_to_item(chest,stack)
                end
            end
        end
    end
end


function mark_deconstruction(event)
    if event.player_index and game.get_player(event.player_index).is_shortcut_toggled(const.shortcut) then
        global.decon_chests[event.entity.unit_number] = event.entity
    end
end

function unmark_deconstruction(event)
    global.decon_chests[event.entity.unit_number] = nil
end

script.on_event(defines.events.on_built_entity,move_to_container,const.entity_filters)
script.on_event(defines.events.on_pre_player_mined_item, move_to_inventory,const.entity_filters)
script.on_event({defines.events.on_lua_shortcut,"packing-tape-pickup"}, toggle_shortcut)
script.on_event(defines.events.on_robot_pre_mined,robot_mined_chest,const.entity_filters)
script.on_event(defines.events.on_cancelled_deconstruction,unmark_deconstruction,const.entity_filters)
script.on_event(defines.events.on_marked_for_deconstruction,mark_deconstruction,const.entity_filters)

-- set it to toggled on when first loading
script.on_event(defines.events.on_player_created, function(event)
    game.get_player(event.player_index).set_shortcut_toggled(const.shortcut,true)
end)

script.on_event(defines.events.on_gui_closed, function(event) 
    if event.gui_type == defines.gui_type.item then
        local item = event.item
        if item and item.valid_for_read and string.find(item.name, const.item_prefix_pattern) then
            unpack_item(game.get_player(event.player_index), item)
        end
    end
end)

local function setup_global()
    global.items = global.items or {}
    global.decon_chests = global.decon_chests or {}
end

script.on_init(setup_global)

script.on_configuration_changed(setup_global)